#!/usr/bin/env bash
# Trim Kubernetes/Docker container logs in place, keeping last N lines.
# Default: keep last 1000 lines.

set -euo pipefail

KEEP_LINES="${KEEP_LINES:-1000}"
DRY_RUN=0

# Common log locations
PATHS=(
  "/var/lib/docker/containers"   # Docker json-file logs
  "/var/log/containers"          # K8s symlinks
  "/var/log/pods"                # K8s pod logs
)

usage() {
  cat <<EOF
Usage: sudo bash $0 [--lines N] [--dry-run] [--path DIR ...]
  --lines N     Keep last N lines (default: ${KEEP_LINES})
  --dry-run     Show what would change, do not modify files
  --path DIR    Add/override paths to scan (can be repeated)
Env: KEEP_LINES can also be set.
EOF
}

# Parse args
CUSTOM_PATHS=()
while [[ $# -gt 0 ]]; do
  case "$1" in
    --lines)
      KEEP_LINES="$2"; shift 2;;
    --dry-run)
      DRY_RUN=1; shift;;
    --path)
      CUSTOM_PATHS+=("$2"); shift 2;;
    -h|--help)
      usage; exit 0;;
    *)
      echo "Unknown arg: $1"; usage; exit 1;;
  esac
done

if [[ ${#CUSTOM_PATHS[@]} -gt 0 ]]; then
  PATHS=("${CUSTOM_PATHS[@]}")
fi

if [[ $EUID -ne 0 ]]; then
  echo "Please run as root (sudo)."; exit 1
fi

echo "Scanning paths: ${PATHS[*]}"
echo "Keeping last ${KEEP_LINES} lines per log."
[[ $DRY_RUN -eq 1 ]] && echo "DRY RUN mode—no changes will be made."

# Collect candidate files
mapfile -t FILES < <(
  for d in "${PATHS[@]}"; do
    [[ -d "$d" ]] || continue
    # Typical Docker/K8s log patterns
    find "$d" -type f \( -name "*-json.log" -o -name "*.log" \) 2>/dev/null
  done | sort -u
)

total_before=0
total_after=0
trimmed_count=0

for f in "${FILES[@]}"; do
  [[ -f "$f" ]] || continue

  # Skip tiny files quickly
  # (If line count <= KEEP_LINES, skip)
  # Note: For speed on huge files, `wc -l` is still fine here; we avoid parsing content.
  lines=$(wc -l < "$f" || echo 0)
  if (( lines <= KEEP_LINES )); then
    continue
  fi

  size_before=$(stat -c%s "$f" 2>/dev/null || echo 0)

  if [[ $DRY_RUN -eq 1 ]]; then
    echo "[DRY] Would trim: $f  (lines: ${lines} -> ${KEEP_LINES}, size: ${size_before} bytes -> ~unknown)"
    ((trimmed_count++))
    ((total_before+=size_before))
    continue
  fi

  tmp=$(mktemp)
  # Capture tail first to avoid losing "last chunk" during truncate
  # copytruncate approach: we keep inode by truncating then writing back
  if tail -n "${KEEP_LINES}" "$f" > "$tmp" 2>/dev/null; then
    # Optional: brief sync to reduce race windows
    sync

    # Preserve mode/owner; we will truncate & append (same inode)
    : > "$f"                    # truncate in place (keeps inode)
    cat "$tmp" >> "$f"

    rm -f "$tmp"

    size_after=$(stat -c%s "$f" 2>/dev/null || echo 0)
    echo "Trimmed: $f  (lines: ${lines} -> ${KEEP_LINES}, size: ${size_before} -> ${size_after} bytes)"
    ((trimmed_count++))
    ((total_before+=size_before))
    ((total_after+=size_after))
  else
    echo "WARN: Failed to read $f; skipping."
    rm -f "$tmp" || true
  fi
done

if [[ $DRY_RUN -eq 1 ]]; then
  echo "DRY summary: files to trim: ${trimmed_count}, total pre-trim size: ${total_before} bytes"
else
  echo "Summary: trimmed ${trimmed_count} files."
  if (( trimmed_count > 0 )); then
    echo "Approx freed: $(( total_before - total_after )) bytes"
  fi
fi
